/*  Fanny Xu, CS812, Phase 3
 *
 *  encodeExpr.c - code generation for expressions
 *
 */

#include "encode.h"
#include "message.h"
#include "analyze.h"

/*  70% level  */

/* This function does the encoding for the binary '+' operator.
 */
void encodeAdd(BinopTree b)
{
  TypeTag myType;
  
  /* error checkings */
  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeAdd\n");
  if( !b->left || !b->right)
    bugT( (Tree)b, "NULL child pointer passed to encodeAdd\n");

  b->expn.reg = setReg();
  
  myType = typeQuery(b->expn.type);
  if( myType == TYERROR ) return; /* should never happen*/
  switch ( myType ){
  case TYFLOAT:
  {
    emit("\tflds\t(%%rsp)");
    emit_stack_shrink(8);
    emit("\tflds\t(%%rsp)");
    emit("\tfaddp");
    emit("\tfstps\t(%%rsp)");
    return;
  }
  
  case TYDOUBLE:
  {
    emit("\tfldl\t(%%rsp)");
    emit_stack_shrink(8);
    emit("\tfldl\t(%%rsp)");
    emit("\tfaddp");
    emit("\tfstpl\t(%%rsp)");
    return;
  }
  
  case TYLONGDOUBLE:
  {
    emit("\tfldt\t(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfldt\t(%%rsp)");
    emit("\tfaddp");
    emit("\tfstpt\t(%%rsp)");
    return;
  }

  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rdx");
    emit_pop_reg("rax");
    emit("\taddl\t%%edx,%%eax");
    emit_push_reg("rax");
    return;
  }

  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  {
    emit_pop_reg("rdx");
    emit_pop_reg("rax");
    emit("\taddq\t%%rdx,%%rax");
    emit_push_reg("rax");
    return;
  }

  /* rel start */
  case TYPOINTER:
  {
    emit_pop_reg("rdx");
    emit_pop_reg("rax");
    emit("\taddq\t%%rdx,%%rax");
    emit_push_reg("rax");
    return;
  }
  /* rel end */

  default:
    bugT((Tree)b, "Invalid tree type in encodeAdd() %d", myType);
 }
  return;
}

/* This function does the encoding for the binary '-' operator.
 */
void encodeSub(BinopTree b)
{
  TypeTag myType;

  /* error checkings */
  if( !b )
    bugT( (Tree)b, "Null pointer passed to encodeSub\n");
  if( !b->left || !b->right )
    bugT( (Tree)b, "Null child pointer passed to encodeSub\n");

  b->expn.reg = setReg();
  myType = typeQuery(b->expn.type);
  if( myType == TYERROR) return;
  switch( myType ){
  case TYFLOAT:
  {
    emit("\tflds\t(%%rsp)");
    emit_stack_shrink(8);
    emit("\tflds\t(%%rsp)");
    emit("\tfsubp");
    emit("\tfstps\t(%%rsp)");
    return;
  }
  
  case TYDOUBLE:
  {
    emit("\tfldl\t(%%rsp)");
    emit_stack_shrink(8);
    emit("\tfldl\t(%%rsp)");
    emit("\tfsubp");
    emit("\tfstpl\t(%%rsp)");
    return;
  }
  
  case TYLONGDOUBLE:
  {
    emit("\tfldt\t(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfldt\t(%%rsp)");
    emit("\tfsubp");
    emit("\tfstpt\t(%%rsp)");
    return;
  }
  
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rdx");
    emit_pop_reg("rax");
    emit("\tsubl\t%%edx, %%eax");
    emit_push_reg("rax");
    return;
  } 

  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  {
    emit_pop_reg("rdx");
    emit_pop_reg("rax");
    emit("\tsubq\t%%rdx, %%rax");
    emit_push_reg("rax");
    return;
  }

  case TYPOINTER:
  {
    emit_pop_reg("rdx");
    emit_pop_reg("rax");
    emit("\tsubq\t%%rdx, %%rax");
    emit_push_reg("rax");
    return;
  }
 
  default:
    bugT((Tree)b, "Invalid tree type in encodeSub()");
 }
  return;
}

/* This function does the encoding for the binary operator '/'.
 */
void encodeDiv(BinopTree b)
{
  TypeTag myType;

  /* error checkings */
  if( !b )
    bugT( (Tree)b, "Null pointer passed to encodeDiv\n");
  if( !b->left || !b->right )
    bugT( (Tree)b, "Null child pointer passed to encodeDiv\n");

  b->expn.reg = setReg();
  myType = typeQuery(b->expn.type);
  if( myType == TYERROR) return;
  switch( myType ){
  case TYFLOAT:
  {
    emit("\tflds\t(%%rsp)");
    emit_stack_shrink(8);
    emit("\tflds\t(%%rsp)");
    emit("\tfdivp");
    emit("\tfstps\t(%%rsp)");
    return;
  }
  
  case TYDOUBLE:
  {
    emit("\tfldl\t(%%rsp)");
    emit_stack_shrink(8);
    emit("\tfldl\t(%%rsp)");
    emit("\tfdivp");
    emit("\tfstpl\t(%%rsp)");
    return;
  }
  
  case TYLONGDOUBLE:
  {
    emit("\tfldt\t(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfldt\t(%%rsp)");
    emit("\tfdivp");
    emit("\tfstpt\t(%%rsp)");
    return;
  }
  
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYSIGNEDCHAR:
  { 
    emit_pop_reg("rcx");
    emit_pop_reg("rax");
    emit("\tcltd");
    emit("\tidivl\t%%ecx,%%eax");
    emit_push_reg("rax");
    return;
  }
 
  case TYSIGNEDLONGINT:
  {
    emit_pop_reg("rcx");
    emit_pop_reg("rax");
    emit("\tcltd");
    emit("\tidivq\t%%rcx,%%rax");
    emit_push_reg("rax");
    return;
  }

  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  {
    emit_pop_reg("rcx");
    emit_pop_reg("rax");
    emit("\tmovq\t$0, %%rdx");
    emit("\tdivl\t%%ecx, %%eax");
    emit_push_reg("rax");
    return;
  } 
  
  case TYUNSIGNEDLONGINT:
  {
    emit_pop_reg("rcx");
    emit_pop_reg("rax");
    emit("\tmovq\t$0, %%rdx");
    emit("\tdivq\t%%rcx, %%rax");
    emit_push_reg("rax");
    return;
  }

  default: 
    bugT((Tree)b, "Invalid tree type in encodeDiv()");
 }

  return;
}

/* This function does the encoding for the binary operator '*'.
 */
void encodeMult(BinopTree b)
{
  TypeTag myType;

  /* error checkings */
  if( !b )
    bugT( (Tree)b, "Null pointer passed to encodeSub\n");
  if( !b->left || !b->right )
    bugT( (Tree)b, "Null child pointer passed to encodeSub\n");

  b->expn.reg = setReg();
  myType = typeQuery(b->expn.type);
  if( myType == TYERROR) return;
  switch( myType ){
  case TYFLOAT:
  {
    emit("\tflds\t(%%rsp)");
    emit_stack_shrink(8);
    emit("\tflds\t(%%rsp)");
    emit("\tfmulp");
    emit("\tfstps\t(%%rsp)");
    return;
  }
  
  case TYDOUBLE:
  {
    emit("\tfldl\t(%%rsp)");
    emit_stack_shrink(8);
    emit("\tfldl\t(%%rsp)");
    emit("\tfmulp");
    emit("\tfstpl\t(%%rsp)");
    return;
  }
  
  case TYLONGDOUBLE:
  {
    emit("\tfldt\t(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfldt\t(%%rsp)");
    emit("\tfmulp");
    emit("\tfstpt\t(%%rsp)");
    return;
  }
  
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rdx");
    emit_pop_reg("rax");
    emit("\timull\t%%edx, %%eax");
    emit_push_reg("rax");
    return;
  } 

  case TYUNSIGNEDLONGINT:
  case TYSIGNEDLONGINT:
  {
    emit_pop_reg("rdx");
    emit_pop_reg("rax");
    emit("\timulq\t%%rdx, %%rax");
    emit_push_reg("rax");
    return;
  }

  default:
    bugT( (Tree)b, "Invalid tree type in encodeMult()");
 }

  return;
}

/* This function does the encoding for the binary '%' operator.
 */
void encodeMod(BinopTree b)
{
  TypeTag myType;

  /* error checkings */
  if( !b )
    bugT( (Tree)b, "Null pointer passed to encodeSub\n");
  if( !b->left || !b->right )
    bugT( (Tree)b, "Null child pointer passed to encodeSub\n");
  
  b->expn.reg = setReg();
  myType = typeQuery(b->expn.type);
  if( myType == TYERROR) return;
  switch( myType ){
  case TYSIGNEDLONGINT:
  {
    emit_pop_reg("rcx");
    emit_pop_reg("rax");
    emit("\tcltd");
    emit("\tidivq\t%%rcx, %%rax");
    emit_push_reg("rdx");
    return;
  }

  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rcx");
    emit_pop_reg("rax");
    emit("\tcltd");
    emit("\tidivl\t%%ecx, %%eax");
    emit_push_reg("rdx");
    return;
  }

  case TYUNSIGNEDLONGINT:
  {
    emit_pop_reg("rcx");
    emit_pop_reg("rax");
    emit("\tmovq\t$0, %%rdx");
    emit("\tdivq\t%%rcx, %%rax");
    emit_push_reg("rdx");
    return;
  }

  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  {
    emit_pop_reg("rcx");
    emit_pop_reg("rax");
    emit("\tmovq\t$0, %%rdx");
    emit("\tdivl\t%%ecx, %%eax");
    emit_push_reg("rdx");
    return;
  }

  default:
    break; 
 }
}

/* Dummy function for encoding '*='. It is never called.
 */
void encodeMultAssign(BinopTree b)
{
  b = NULL; /* defeat warning */
  info("encodeMultAssign: You are not supposed to get here!");
  return;
}

/* Dummy function for encoding '/='. It is never called.
 */
void encodeDivAssign(BinopTree b)
{
  b = NULL; /* defeat warning */
  info("encodeDivAssign: You are not supposed to get here!");
  return;
}

/* Dummy function for encoding '%='. It is never called.
 */
void encodeModAssign(BinopTree b)
{
  b = NULL; /* defeat warning */
  info("encodeModAssign: You are not supposed to get here!");
  return;
}

/* Dummy function for encoding '+='. It is never called.
 */
void encodeAddAssign(BinopTree b)
{
  b = NULL; /* defeat warning */
  info("encodeAddAssign: You are not supposed to get here!");
  return;
}

/* Dummy function for encoding '-='. It is never called.
 */
void encodeSubAssign(BinopTree b)
{
  b = NULL; /* defeat warning */
  info("encodeSubAssign: You are not supposed to get here!");
  return;
}

/* This function does the encoding for the unary '-' operator.
 */
void encodeUsub(UnopTree u)
{
  TypeTag myType;

  /* error checkings */
  if( !u )
    bugT( (Tree)u, "Null pointer passed to encodeUsub\n");
  if( !u->left )
    bugT( (Tree)u, "Null child pointer passed to encodeUsub\n");

  u->expn.reg = setReg();
  myType = typeQuery(u->expn.type);
  if( myType == TYERROR ) return;
  switch( myType ){
  case TYFLOAT:
  {
    emit_pop_reg("rax");
    emit("\txorl\t$0x80000000, %%eax");
    emit_push_reg("rax");
    return;
  }  
  case TYDOUBLE:
  {
    /* no 64-bit immediate values */
    emit_pop_reg("rax");
    emit_push_const(0);
    emit("\tmovl\t$0x80000000, 4(%%rsp)");
    emit_pop_reg("rdx");
    emit("\txorq\t%%rdx, %%rax");
    emit_push_reg("rax");
    return;
  }    
  case TYLONGDOUBLE:
  {
    emit_pop_reg("edx");
    emit_pop_reg("eax");
    emit("\txorl\t$0x8000, %%eax");
    emit_push_reg("eax");
    emit_push_reg("edx");
    return;
  }    

  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  {
    emit_pop_reg("rax");
    emit("\tnegq\t%%rax");
    emit_push_reg("rax");
    return;
  }

  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rax");
    emit("\tnegl\t%%eax");
    emit_push_reg("rax");
    return;
  }

  default:
    bugT((Tree)u, "Invalid tree type in encodeUsub()\n");
 }
 return; 
}


/*  80% level  */

/* Dummy funtion to encode the unary operator '!'.
 * It is never called.
 */
void encodeLnot(UnopTree b)
{ 
  b = NULL; /* defeat warning */
  info("encodeLnot: You are not supposed to get here!");
  return; 
} 

/* This function does the encoding for the binary operator '<'.
 */
void encodeLt(BinopTree b)
{
  TypeTag myType;

  /* error checkings */ 
  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeLt\n");
  if( !b->left || !b->right)
    bugT( (Tree)b, "NULL child pointer passed to encodeLt\n");

  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( myType == TYERROR ) return; 
  switch ( myType ){
  case TYFLOAT:
  {
    emit("\tflds\t8(%%rsp)");
    emit("\tflds\t(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$69, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  
  case TYDOUBLE:
  {
    emit("\tfldl\t8(%%rsp)");
    emit("\tfldl\t(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$69, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  
  case TYLONGDOUBLE:
  {
    emit("\tfldt\t16(%%rsp)");
    emit("\tfldt\t(%%rsp)");
    emit_stack_shrink(32);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$69, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }

  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpl\t%%eax,%%edx");
    if( myType == TYSIGNEDSHORTINT || myType == TYSIGNEDINT ||
		myType == TYSIGNEDCHAR )
      emit("\tsetl\t%%al");
    else
      emit("\tsetb\t%%al"); 
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  case TYPOINTER:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpq\t%%rax,%%rdx");
    if( myType == TYSIGNEDLONGINT )
      emit("\tsetl\t%%al");
    else
      emit("\tsetb\t%%al"); 
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  default:
    bugT((Tree)b, "Invalid tree type in encodeLt()");
 }
 return;
}

/* This function does the encoding for the binary operator '>'.
 */
void encodeGt(BinopTree b)
{
  TypeTag myType;
  
  /* error checkings */
  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeGt\n");
  if( !b->left || !b->right)
    bugT( (Tree)b, "NULL child pointer passed to encodeGt\n");

  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( myType == TYERROR ) return; 
  switch ( myType ){
  case TYFLOAT:
  {
    emit("\tflds\t(%%rsp)");
    emit("\tflds\t8(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$69, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  
  case TYDOUBLE:
  {
    emit("\tfldl\t(%%rsp)");
    emit("\tfldl\t8(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$69, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  
  case TYLONGDOUBLE:
  {
    emit("\tfldt\t(%%rsp)");
    emit("\tfldt\t16(%%rsp)");
    emit_stack_shrink(32);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$69, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpl\t%%eax,%%edx");
    if( myType == TYSIGNEDSHORTINT || myType == TYSIGNEDINT ||
		myType == TYSIGNEDCHAR )
      emit("\tsetg\t%%al");
    else
      emit("\tseta\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYUNSIGNEDLONGINT:
  case TYSIGNEDLONGINT:
  case TYPOINTER:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpq\t%%rax,%%rdx");
    if( myType == TYSIGNEDLONGINT )
      emit("\tsetg\t%%al");
    else
      emit("\tseta\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  default:
    bugT((Tree)b, "Invalid tree type in encodeGt()"); 
 }
 return;
}

/* This function does the encoding for the binary operator '<='.
 */
void encodeLte(BinopTree b)
{
  TypeTag myType;
  
  /* error checkings */
  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeLte\n");
  if( !b->left || !b->right)
    bugT( (Tree)b, "NULL child pointer passed to encodeLte\n");

  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( myType == TYERROR ) return; 
  switch ( myType ){
  case TYFLOAT:
  {
    emit("\tflds\t8(%%rsp)");
    emit("\tflds\t(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$5, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYDOUBLE:
  {
    emit("\tfldl\t8(%%rsp)");
    emit("\tfldl\t(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$5, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  
  case TYLONGDOUBLE:
  {
    emit("\tfldt\t16(%%rsp)");
    emit("\tfldt\t(%%rsp)");
    emit_stack_shrink(32);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$5, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpl\t%%eax,%%edx");
    if( myType == TYSIGNEDSHORTINT || myType == TYSIGNEDINT ||
		myType == TYSIGNEDCHAR )
      emit("\tsetle\t%%al");
    else
      emit("\tsetbe\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  case TYPOINTER:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpq\t%%rax,%%rdx");
    if( myType == TYSIGNEDLONGINT )
      emit("\tsetle\t%%al");
    else
      emit("\tsetbe\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  default:
    bugT((Tree)b, "Invalid tree type in encodeLte()");
 }
 return;
}

/* This function does the encoding for the binary operator '>='.
 */
void encodeGte(BinopTree b)
{

  TypeTag myType;
 
  /* error checkings */
  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeGte\n");
  if( !b->left || !b->right)
    bugT( (Tree)b, "NULL child pointer passed to encodeGte\n");

  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( myType == TYERROR ) return; 
  switch ( myType ){
  case TYFLOAT:
  {
    emit("\tflds\t(%%rsp)");
    emit("\tflds\t8(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$5, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYDOUBLE:
  {
    emit("\tfldl\t(%%rsp)");
    emit("\tfldl\t8(%%rsp)");
	emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$5, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbl\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYLONGDOUBLE:
  {
    emit("\tfldt\t(%%rsp)");
    emit("\tfldt\t16(%%rsp)");
    emit_stack_shrink(32);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    emit("\ttestb\t$5, %%ah");
    emit("\tsete\t%%al");
    emit("\tmovzbl\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }

  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpl\t%%eax,%%edx");
    if( myType == TYSIGNEDSHORTINT || myType == TYSIGNEDINT ||
        myType == TYSIGNEDCHAR )
      emit("\tsetge\t%%al");
    else
      emit("\tsetae\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  case TYPOINTER:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpq\t%%rax,%%rdx");
    if( myType == TYSIGNEDLONGINT || myType == TYSIGNEDSHORTINT ||
        myType == TYSIGNEDINT     || myType == TYSIGNEDCHAR )
      emit("\tsetge\t%%al");
    else
      emit("\tsetae\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  default:
    bugT( (Tree)b, "Invalid tree type in encodeGte()");
 }
 return;
}

/* This function does the encoding for the binary operator '=='.
 */
void encodeEql(BinopTree b)
{
  TypeTag myType;
  
  /* error checkings */
  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeEql\n");
  if( !b->left || !b->right)
    bugT( (Tree)b, "NULL child pointer passed to encodeEql\n");

  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( myType == TYERROR ) return; 
  switch ( myType ){
  case TYFLOAT:
  {
    emit("\tflds\t(%%rsp)");
    emit("\tflds\t8(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    
    emit("\tandb\t$69, %%ah");
    emit("\tcmpb\t$64, %%ah");
    emit("\tsete\t%%al");

    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYDOUBLE:
  {
    emit("\tfldl\t(%%rsp)");
    emit("\tfldl\t8(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");

    emit("\tandb\t$69, %%ah");
    emit("\tcmpb\t$64, %%ah");
    emit("\tsete\t%%al");

    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYLONGDOUBLE:
  {
    emit("\tfldt\t(%%esp)");
    emit("\tfldt\t16(%%esp)");
    emit_stack_shrink(32);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");

    emit("\tandb\t$69, %%ah");
    emit("\tcmpb\t$64, %%ah");
    emit("\tsete\t%%al");

    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYSIGNEDCHAR:
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpl\t%%eax,%%edx");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  case TYPOINTER:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpq\t%%rax,%%rdx");
    emit("\tsete\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  default:
    bugT((Tree)b, "Invalid tree type in encodeEql()");
 }
 return;
}

/* This function does the encoding for the binary operator '!='.
 */
void encodeNeql(BinopTree b)
{
  TypeTag myType;
  
  /* error checkings */
  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeNeql\n");
  if( !b->left || !b->right)
    bugT( (Tree)b, "NULL child pointer passed to encodeNeql\n");

  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( myType == TYERROR ) return; 
  switch ( myType ){
  case TYFLOAT:
  {
    emit("\tflds\t(%%rsp)");
    emit("\tflds\t8(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");
    
    emit("\tandb\t$69, %%ah");
    emit("\txorb\t$64, %%ah");
    emit("\tsetne\t%%al");

    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYDOUBLE:
  {
    emit("\tfldl\t(%%rsp)");
    emit("\tfldl\t8(%%rsp)");
    emit_stack_shrink(16);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");

    emit("\tandb\t$69, %%ah");
    emit("\txorb\t$64, %%ah");
    emit("\tsetne\t%%al");

    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYLONGDOUBLE:
  {
    emit("\tfldt\t(%%rsp)");
    emit("\tfldt\t16(%%rsp)");
    emit_stack_shrink(32);
    emit("\tfucompp");
    emit("\tfnstsw\t%%ax");

    emit("\tandb\t$69, %%ah");
    emit("\txorb\t$64, %%ah");
    emit("\tsetne\t%%al");

    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpl\t%%eax,%%edx");
    emit("\tsetne\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }  
  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  case TYPOINTER:
  {
    emit_pop_reg("rax");
    emit_pop_reg("rdx");
    emit("\tcmpq\t%%rax,%%rdx");
    emit("\tsetne\t%%al");
    emit("\tmovzbq\t%%al, %%rax");
    emit_push_reg("rax");
    return;
  }
  default:
    bugT((Tree)b, "Invalid tree type to encodeNeql()");
 }
 return;
}

/* This function does the encoding for the binary operator '&&'.
 */
void encodeLand(BinopTree b)
{
  TypeTag myType;
  Type newType;
  UnopTree uTree;
  int lab1, lab2;

  /* error checkings */
  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeLand\n");
  if( !b->left || !b->right)
    bugT( (Tree)b, "NULL child pointer passed to encodeLand\n");

  lab1 = genLabel();
  lab2 = genLabel();
  
  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( typeQuery(((ExpTree)b->left)->type) == TYERROR ||
      typeQuery(((ExpTree)b->right)->type) == TYERROR )
  { return;
  }
 
  /* encodes the left operand first */
  switch ( myType ){
  case TYFLOAT:
  case TYDOUBLE:
  case TYLONGDOUBLE:
  {
    newType = typeBuildBasic( myType, NO_QUAL );
    uTree = (UnopTree)newUnop( LNOT_OP, newType, b->left);
    encode( (Tree)analyzeLnot( uTree ));
    
	emit_pop_reg("rax");
    emit("\tcmpq\t$0, %%rax");
    emit("\tjne\tL$%d", lab1);

    myType = typeQuery( ((ExpTree)b->right)->type);
    break;
  }

  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    encode( (Tree)b->left );
	emit_pop_reg("rax");
    emit("\tcmpl\t$0, %%eax");
    emit("\tje\tL$%d", lab1);/* if it's zero, && fails*/
 
    myType = typeQuery( ((ExpTree)b->right)->type);   
    break;
  }
  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  case TYPOINTER:
  {
    encode( (Tree)b->left );
	emit_pop_reg("rax");
    emit("\tcmpq\t$0, %%rax");
    emit("\tje\tL$%d", lab1);/* if it's zero, && fails*/
 
    myType = typeQuery( ((ExpTree)b->right)->type);   
    break;
  }
  default:
    bugT((Tree)b, "Invalid left subtree type in encodeLand()");
 }

  /* encodes the right operand */
  switch( myType ){
  case TYFLOAT:
  case TYDOUBLE:
  case TYLONGDOUBLE:
  {
    newType = typeBuildBasic( myType, NO_QUAL);
    uTree = (UnopTree)newUnop( LNOT_OP, newType, b->right);
    encode( (Tree)analyzeLnot( uTree));
    
	emit_pop_reg("rax");
    emit("\tcmpq\t$0, %%rax");
    emit("\tjne\tL$%d", lab1);
	emit_push_const(1);
    emit("\tjmp\tL$%d", lab2);
    emit("L$%d:", lab1 );
    emit("\tpushq\t$0");   /* don't count the constant on stack twice! */
    emit("L$%d:", lab2 );
    return;

  }
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
   encode( (Tree)b->right );
   emit_pop_reg("rax");
   emit("\tcmpl\t$0, %%eax");
   emit("\tje\tL$%d", lab1);
   emit_push_const(1);
   emit("\tjmp\tL$%d", lab2);
   emit("L$%d:", lab1 );
   emit("\tpushq\t$0");   /* don't count the constant on stack twice! */
   emit("L$%d:", lab2 );
   return;
  }
  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  case TYPOINTER:
  {
   encode( (Tree)b->right );
   emit_pop_reg("rax");
   emit("\tcmpq\t$0, %%rax");
   emit("\tje\tL$%d", lab1);
   emit_push_const(1);
   emit("\tjmp\tL$%d", lab2);
   emit("L$%d:", lab1 );
   emit("\tpushq\t$0");   /* don't count the constant on stack twice! */
   emit("L$%d:", lab2 );
   return;
  }
  default:
    bugT((Tree)b, "Invalid right subtree type in encodeLand()");
 }
 return;
}

/* This function does the encoding for the binary '||'.
 */ 
void encodeLor(BinopTree b)
{
  TypeTag myType;
  Type newType;
  UnopTree uTree;
  int lab1, lab2;

  /* error checkings */
  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeLor\n");
  if( !b->left || !b->right)
    bugT( (Tree)b, "NULL child pointer passed to encodeLor\n");

  lab1 = genLabel();
  lab2 = genLabel();
  
  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( typeQuery(((ExpTree)b->left)->type ) == TYERROR ||
      typeQuery(((ExpTree)b->right)->type ) == TYERROR )
    return; 

  switch ( myType ){
  case TYFLOAT:
  case TYDOUBLE:
  case TYLONGDOUBLE:
  {
    newType = typeBuildBasic( myType, NO_QUAL );
    uTree = (UnopTree)newUnop( LNOT_OP, newType, b->left);
    encode( (Tree)analyzeLnot( uTree ));
    
    emit_pop_reg("rax");
    emit("\tcmpq\t$1, %%rax");
    emit("\tjne\tL$%d", lab1);
    
    myType = typeQuery( ((ExpTree)b->right)->type );
    break;
  }
  
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:  
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    encode((Tree)b->left);
    emit_pop_reg("rax");
    emit("\tcmpl\t$0, %%eax");
    emit("\tjne\tL$%d", lab1);
    
    myType = typeQuery( ((ExpTree)b->right)->type );
    break;

  }
  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  case TYPOINTER:
  {
    encode((Tree)b->left);
    emit_pop_reg("rax");
    emit("\tcmpq\t$0, %%rax");
    emit("\tjne\tL$%d", lab1);
    
    myType = typeQuery( ((ExpTree)b->right)->type );
    break;

  }
  default:
    bugT( (Tree)b, "Invalid left subtree type in encodeLor()");
 }

  switch( myType ){
  case TYFLOAT:
  case TYDOUBLE:
  case TYLONGDOUBLE:
  {
    newType = typeBuildBasic( myType, NO_QUAL );
    uTree = (UnopTree)newUnop( LNOT_OP, newType, b->right);
    encode( (Tree)analyzeLnot( uTree));
    
    emit_pop_reg("rax");
    emit("\tcmpq\t$1, %%rax");
    emit("\tjne\tL$%d", lab1);
    emit_push_const(0);
    emit("\tjmp\tL$%d", lab2);
    emit("L$%d:", lab1 );
    emit("\tpushq\t$1");   /* don't count the constant on stack twice! */
    emit("L$%d:", lab2 );
    return;
  }


  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
   encode((Tree)b->right );
   emit_pop_reg("rax");
   emit("\tcmpl\t$0, %%eax");
   emit("\tjne\tL$%d", lab1);
   emit_push_const(0);
   emit("\tjmp\tL$%d", lab2);
   emit("L$%d:", lab1 );
   emit("\tpushq\t$1");   /* don't count the constant on stack twice! */
   emit("L$%d:", lab2 );
   return;
  }
  case TYUNSIGNEDLONGINT:
  case TYSIGNEDLONGINT:
  case TYPOINTER:
  {
   encode((Tree)b->right );
   emit_pop_reg("rax");
   emit("\tcmpq\t$0, %%rax");
   emit("\tjne\tL$%d", lab1);
   emit_push_const(0);
   emit("\tjmp\tL$%d", lab2);
   emit("L$%d:", lab1 );
   emit("\tpushq\t$1");   /* don't count the constant on stack twice! */
   emit("L$%d:", lab2 );
   return;
  }
  default:
   bugT( (Tree)b, "Invalid right subtree type in encodeLor()");
 }

  return;
}

/*  90% level  */

/* This function does encoding for the unary operator '~'.
 */
void encodeComplmt(UnopTree u)
{
  TypeTag myType;

  if( !u )
    bugT( (Tree)u, "NULL pointer passed to encodeComplmt\n");
  if( !u->left)
    bugT( (Tree)u, "NULL child pointer passed to encodeComplmt\n");
  
  u->expn.reg = setReg();
  myType = typeQuery(((ExpTree)u->left)->type);
  if( myType == TYERROR ) return; 
  switch ( myType ){
  case TYSIGNEDLONGINT:
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDLONGINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rax");
    emit("\tnotq\t%%rax"); 
    emit_push_reg("rax");
   return;
  }
  default:
    bugT( (Tree)u, "Invalid tree type in encodeComplmt()");
 }
  return;
}

/* This function does encoding for the binary operator '<<'.
 */
void encodeLshift(BinopTree b)
{
  TypeTag myType;

  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeLshift\n");
  if( !(b->left && b->right))
    bugT( (Tree)b, "NULL child pointer passed to encodeLshift\n");
  
  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( myType == TYERROR ) return; 
  switch ( myType ){
  case TYSIGNEDLONGINT:
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDLONGINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit("\tmovb\t(%%rsp), %%cl");
    emit_stack_shrink(8);
    emit_pop_reg("rax");
    emit("\tsalq\t%%cl, %%rax");
	emit_push_reg("rax");
    return;
  }
  default:
    bugT((Tree)b, "invalid tree type in encodeLshift()");
 }
  return;
}

/* This function does encoding for the binary operator '>>'.
 */
void encodeRshift(BinopTree b)
{
  TypeTag myType;

  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeRshift\n");
  if( !(b->left && b->right))
    bugT( (Tree)b, "NULL child pointer passed to encodeRshift\n");
  
  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( myType == TYERROR ) return; 
  switch ( myType ){
  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  {
    emit("\tmovb\t(%%rsp), %%cl");
    emit_stack_shrink(8);
    emit_pop_reg("rax");
    if(myType == TYUNSIGNEDLONGINT)
      emit("\tsarq\t%%cl, %%rax");
    else
      emit("\tshrq\t%%cl, %%rax");
    emit_push_reg("rax");
    return;
  }

  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit("\tmovb\t(%%rsp), %%cl");
    emit_stack_shrink(8);
    emit_pop_reg("rax");
    if( myType == TYSIGNEDSHORTINT || myType == TYSIGNEDINT ||
        myType == TYSIGNEDCHAR)
      emit("\tsarl\t%%cl, %%eax");
    else
      emit("\tshrl\t%%cl, %%eax");
    emit_push_reg("rax");
    return;
  }
  default:
    bugT( (Tree)b, "Invalid tree type in encodeRshift()");
 }
  return;
}

/* This function does encoding for the binary operator '&'.
 */
void encodeAndd(BinopTree b)
{
  TypeTag myType;

  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeAndd\n");
  if( !(b->left && b->right))
    bugT( (Tree)b, "NULL child pointer passed to encodeAndd\n");
  
  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( myType == TYERROR ) return;
  switch ( myType ){
  case TYSIGNEDLONGINT:
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDLONGINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rdx");
    emit_pop_reg("rax");
    emit("\tandq\t%%rdx, %%rax");
    emit_push_reg("rax");
    return;
  }
  default:
    bugT( (Tree)b, "Invalid tree type in encodeAndd()");
 }
 return;
}

/* This function does encoding for the binary operator '^'.
 */
void encodeXorr(BinopTree b)
{
  TypeTag myType;

  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeXorr\n");
  if( !(b->left && b->right))
    bugT( (Tree)b, "NULL child pointer passed to encodeXorr\n");
  
  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( myType == TYERROR ) return; 
  switch ( myType ){
  case TYSIGNEDLONGINT:
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDLONGINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rdx");
    emit_pop_reg("rax");
    emit("\txorq\t%%rdx, %%rax");
    emit_push_reg("rax");
    return;
  }
  default:
    bugT( (Tree)b, "Invalid tree type in encodeXorr()\n");
 }
 return;
}

/* This function does encoding for the binary operator '|'.
 */
void encodeOrr(BinopTree b)
{
  TypeTag myType;

  if( !b )
    bugT( (Tree)b, "NULL pointer passed to encodeOrr\n");
  if( !(b->left && b->right))
    bugT( (Tree)b, "NULL child pointer passed to encodeOrr\n");
  
  b->expn.reg = setReg();
  myType = typeQuery(((ExpTree)b->left)->type);
  if( myType == TYERROR ) return; 
  switch ( myType ){
  case TYSIGNEDLONGINT:
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDLONGINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rdx");
    emit_pop_reg("rax");
    emit("\torq\t%%rdx, %%rax");
    emit_push_reg("rax");
    return;
  }
  default:
    bugT((Tree)b, "Invalid tree type in encodeOrr()\n");
 }
  return;
}

/* Dummy function to encode leftAssign. It is never called.
 */
void encodeLeftAssign(BinopTree b)
{
  b = NULL; /* defeat warning */
  info("encodeleftAssign: You are not supposed to get here!");
  return;
}

/* Dummy function to encode righttAssign. It is never called.
 */
void encodeRightAssign(BinopTree b)
{
  b = NULL; /* defeat warning */
  info("encodeRightAssign: You are not supposed to get here!");
  return;
}

/* Dummy function to encode AndAssign. It is never called.
 */
void encodeAndAssign(BinopTree b)
{
  b = NULL; /* defeat warning */
  info("encodeAndAssign: You are not supposed to get here!");
  return;
}

/* Dummy function to encode OrAssign. It is never called.
 */
void encodeOrAssign(BinopTree b)
{
  b = NULL; /* defeat warning */
  info("encodeOrAssign: You are not supposed to get here!");
  return;
}

/* Dummy function to encode XorAssign. It is never called.
 */
void encodeXorAssign(BinopTree b)
{
  b = NULL; /* defeat warning */
  info("encodeXorAssign: You are not supposed to get here!");
  return;
}

/*  100% level  */

/* This function does the encoding for the postfix '++' operator.
 */
void encodeIncra(UnopTree u)
{
  /* error checkings*/
  if(!u)
    bugT((Tree)u, "Null pointer passed to encodeIncra\n");
  if(!(u->left) )
    bugT( (Tree)u, "Null child pointer passed to encodeIncra\n");

  /* getting the address */
  emit("\tmovq\t(%%rsp), %%rcx");

  /* address popped and the value right on the stack*/
  encodeDerefDetail( u->expn.type, (Tree)u);
  
  /* addition */
  switch( typeQuery( u->expn.type) ){
  case TYFLOAT:
  {  
     emit("\tflds\t(%%rsp)");    /* load the value */
     emit("\tfld1");             /* load 1.0 */
     emit("\tfaddp");            /* sum  and pop top f.p */
     emit_push_reg("rcx");       /* push the address on the stack */
     emit_stack_grow(8);         /* reserve 8 bytes for the result*/
     emit("\tfstps\t(%%rsp)");   /* put sum back to the run time stack */

     encodeAssignDetail( u->expn.type, (Tree)u); 
     emit_stack_shrink(8);       /* pop off the value from the assignment */
     break;
  }
  
  case TYDOUBLE:
  {  emit("\tfldl\t(%%rsp)");
     emit_push_reg("rxc");
     emit("\tfld1");
     emit("\tfaddp");
     emit_stack_grow(8);
     emit("\tfstpl\t(%%rsp)");

     encodeAssignDetail( u->expn.type, (Tree)u);
     emit_stack_shrink(8);
     break;
  }
  
  case TYLONGDOUBLE:
  {  emit("\tfldt\t(%%rsp)");
     emit("\tfld1");
     emit("\tfaddp");
     emit_push_reg("rxc");
     emit_stack_grow(16);
     emit("\tfstpt\t(%%rsp)");

     encodeAssignDetail( u->expn.type, (Tree)u );
     emit_stack_shrink(16);
     break;
  }
  
  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  {
     /* keep the value as returned value for postfix ++ */
     emit("\tmovq\t(%%rsp), %%rdx");
     emit("\tmovq\t$1, %%rax");
     emit("\taddq\t%%rdx, %%rax");
     emit_push_reg("rcx");
     emit_push_reg("rax");
     encodeAssignDetail( u->expn.type, (Tree)u );
     emit_stack_shrink(8);
     break;
  }

  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
     /* keep the value as returned value for postfix ++ */
     emit("\tmovq\t(%%rsp), %%rdx");
     emit("\tmovq\t$1, %%rax");
     emit("\taddl\t%%edx, %%eax");
     emit_push_reg("rcx");
     emit_push_reg("rax");
     encodeAssignDetail( u->expn.type, (Tree)u );
     emit_stack_shrink(8);
     break;
  }
  case TYPOINTER:
  {
    TypeQualifier qual;   /* temporary output paramater */
    int size;
    size = computeSize(typeQueryPointer(u->expn.type, &qual));
    emit("\tmovq\t(%%rsp), %%rdx");
    emit("\tmovq\t$%d, %%rax", size);
    emit("\taddq\t%%rdx, %%rax");
    emit_push_reg("rcx");
    emit_push_reg("rax");

    encodeAssignDetail( u->expn.type, (Tree)u );
    emit_stack_shrink(8);
    break;
  }
  default:
     bugT( (Tree)u, "invalid tree type in encodeIncra()\n");
 }
  u->expn.reg = setReg();
  return;
}

/* Dummy function for encoding prefix '++' operator.
 * It never gets called.
 */
void encodeIncrb(UnopTree u )
{
  u = NULL; /* defeat warning */
  info("encodeIncrb: You are not supposed to get here!\n");
  return; 
}

/* This function does the encoding for the prefix '++' operator.
 */
void encodeDecra(UnopTree u)
{
  /* error checkings */
  if( !u )
    bugT( (Tree)u, "Null pointer passed to encodeDecra\n");
  if( !(u->left))
    bugT( (Tree)u, "Null child pointer passed to encodeDecra\n");

  emit("\tmovq\t(%%rsp), %%rcx");/* getting the address */
  /* address popped,the value right on the stack*/
  encodeDerefDetail( u->expn.type, (Tree)u);
  /* addition */
  switch( typeQuery( u->expn.type) ){
  case TYFLOAT:
  {  
     emit("\tflds\t(%%rsp)"); /* load the value */
     emit_push_reg("rcx");
     emit("\tfld1"); /* load 1.0 */
     emit("\tfxch");
     emit("\tfsubp"); /* sum  and pop top f.p */
     emit_stack_grow(8);
     emit("\tfstps\t(%%rsp)");/* put sum back to the run time stack */

     encodeAssignDetail( u->expn.type, (Tree)u);
     emit_stack_shrink(8);
     break;
  }
  case TYDOUBLE:
  {  emit("\tfldl\t(%%rsp)");
     emit_push_reg("rcx");
     emit("\tfld1");
     emit("\tfxch");
     emit("\tfsubp");
     emit_stack_grow(8);
     emit("\tfstpl\t(%%rsp)");

     encodeAssignDetail( u->expn.type, (Tree)u);
     emit_stack_shrink(8);
     break;
  }
  case TYLONGDOUBLE:
  {  emit("\tfldt\t(%%rsp)");
     emit("\tfld1");
     emit("\tfxch");
     emit("\tfsubp");
     emit_push_reg("rcx");
     emit_stack_grow(16);
     emit("\tfstpt\t(%%rsp)");

     encodeAssignDetail( u->expn.type, (Tree)u );
     emit_stack_shrink(16);
     break;
  }

  case TYSIGNEDLONGINT:
  case TYUNSIGNEDLONGINT:
  {
     /* keep the value as returned value for postfix -- */
     emit("\tmovq\t(%%rsp), %%rdx");
     emit("\tmovq\t$1, %%rax");
     emit("\tsubq\t%%rax, %%rdx");
     emit_push_reg("rcx");   /* push addr*/
     emit_push_reg("rdx");   /* push result*/
     encodeAssignDetail( u->expn.type, (Tree)u );
     emit_stack_shrink(8);
     break;
  }

  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
     /* keep the value as returned value for postfix -- */
     emit("\tmovq\t(%%rsp), %%rdx");
     emit("\tmovq\t$1, %%rax");
     emit("\tsubl\t%%eax, %%edx");
     emit_push_reg("rcx");   /* push addr*/
     emit_push_reg("rdx");   /* push result*/
     encodeAssignDetail( u->expn.type, (Tree)u );
     emit_stack_shrink(8);
     break;
  }
  /* rel start */
  case TYPOINTER:
  {
    TypeQualifier qual;   /* temporary output paramater */
    int size;
    size = computeSize(typeQueryPointer(u->expn.type, &qual));
    emit("\tmovq\t(%%rsp), %%rdx");
    emit("\tmovq\t$%d, %%rax", size);
    emit("\tsubq\t%%rax, %%rdx");
    emit_push_reg("rcx");
    emit_push_reg("rdx");
    encodeAssignDetail( u->expn.type, (Tree)u );
    emit_stack_shrink(8);
    break;
  }
  /* rel finish */
  default:
     bugT( (Tree)u, "invalid tree type in encodeDecra()\n");
 }
  u->expn.reg = setReg();
  return;
}

/* Dummy function for encoding the prefix '++' operator.
 */
void encodeDecrb(UnopTree u )
{
  u = NULL; /* defeat warning */
  info("encodeDerb: You are not supposed to get here!\n");
  return;
}

/* Dummy function for encoding the cast operator.
 */
void encodeCast(CastopTree t )
{ 
  t = NULL; /* defeat warning */
  info("encodeCast: You are not supposed to get here!\n");
  return;
}

/*  This function encodes the comma operator.
 */
void encodeComma(BinopTree b)
{
  /* error checkings */
  if( !b )
    bugT( (Tree)b, "Null pointer passed to encodeComma\n");
  if( !b->left || !b->right )
    bugT( (Tree)b, "Null child pointer passed to encodeComma\n");

  /* evaluates the left operand first */
  encode( (Tree)(b->left));  
  
  /* pop the stack in order to have the left operand evaluated as void */
  freeExpressionStatementRegister( (Tree)b->left );
  
  /* evaluates the right operand, leaving the returned value on the stack */
  encode( (Tree)(b->right));

  /* mark the tree */
  b->expn.reg = setReg();
}

/* This function encodes the conditional operator.
 */
void encodeCond(TriopTree t)
{
  TypeTag myType;
  int lab1, lab2;
  int saveStackDepth;
  extern int encodeStackDepth;

  lab1 = genLabel();
  lab2 = genLabel();

  /* error checkings */
  if( !t )
    bugT( (Tree)t, "Null pointer passed to encodeCond\n");
  if( !( t->left && t->middle && t->right ))
    bugT( (Tree)t, "Null child pointer passed to encodeCond\n");

  /*
   * We save the stack depth here because the encoding of the 2nd and
   * third operands leaves the stack off by +8.  Over the entire
   * conditional there is a +8, so we will just basically throw away any
   * stack depth computation happening inside the expressions.
   */
  saveStackDepth = encodeStackDepth;

  /* encode the first operand's exp. tree first, which has a Lnot operator */
  encode( (Tree)t->left );
  
  myType = typeQuery( ((ExpTree)t->left)->type );
  switch( myType ){
  case TYFLOAT:
  case TYDOUBLE:
  case TYLONGDOUBLE:
  case TYSIGNEDLONGINT:
  case TYSIGNEDSHORTINT:
  case TYSIGNEDINT:
  case TYUNSIGNEDLONGINT:
  case TYUNSIGNEDSHORTINT:
  case TYUNSIGNEDINT:
  case TYUNSIGNEDCHAR:
  case TYSIGNEDCHAR:
  {
    emit_pop_reg("rax");
    emit("\tmovq\t$0, %%rdx"); 
    emit("\tcmpq\t%%rax, %%rdx"); 
    emit("\tjne\tL$%d", lab1 ); 
  }
  default:
    break;
 }
  /* encode the second operand's expression tree */
  encode( (Tree)t->middle ); 
  emit("\tjmp\tL$%d", lab2 );
  emit("L$%d:", lab1 );
  /* encode the third operand's expression tree */
  encode( (Tree)t->right ); 
  emit("L$%d:", lab2 );

  encodeStackDepth = saveStackDepth + 8;

  t->expn.reg = setReg();
  return;
}

/*  phase 5   */

/**
 * Encodes the AddrOf '&' operator. Relies on recursive calls from skeleton
 * to other methods to to encode the guts of it, so it actually does nothing. 
**/

void encodeAddrOf(UnopTree u)
{
  /* do nothing */
  u->expn.reg = setReg();
}

/**
 * Encodes the Index '[]' operator. Should never be called.
**/ 

void encodeIndex(BinopTree b)
{
  bugT((Tree)b, "encodeIndex should never be called");
}

/**
 * Encodes the '.' operator.
**/ 

void encodeFieldRef(FieldRefTree f)
{
   int offset;
   offset = computeMemberOffset(f->left->type, f->id);

   emit("\n#\t*************** start encode field ref **********");

   emit_pop_reg("rax");                /* pop base address into rax */
   emit("\taddq\t$%d, %%rax", offset); /* add offset to base addr   */
   emit_push_reg("rax");               /* push new addr onto stack  */
   
   emit("#\t***************  end encode field ref  *********\n");
  
   f->expn.reg = setReg();
}

/**
 * Encodes the '->' operator. Should never be called.
**/

void encodePtr(PtrTree p)
{
  bugT((Tree)p, "encodePtr should never be called");
}
